#include "SSH.h"

#define LLOG(x) // DLOG(x)

namespace Upp {

bool KnownHosts::Add(const String& host, int port, const Info& info, const String& comment)
{
	return Add(Format("[%s]:%d", host, port), info, comment);
}

bool KnownHosts::Add(const String& host, const Info& info, const String& comment)
{
	ASSERT(ssh);
	bool b = hosts &&
		libssh2_knownhost_addc(
			hosts,
			~host,
			NULL,
			~info.key, info.key.GetLength(),
			~comment, comment.GetLength(),
			info.type |
			LIBSSH2_KNOWNHOST_TYPE_PLAIN |
			LIBSSH2_KNOWNHOST_KEYENC_RAW,
			NULL
		) == 0;
	return b ? b : Error();
}

bool KnownHosts::Remove(SshHost* host)
{
	ASSERT(ssh);
	auto b = hosts && libssh2_knownhost_del(hosts, host) == 0;
	return b ? b : Error();
}

bool KnownHosts::Load(const String& filename)
{
	ASSERT(ssh);
	file_path = filename;
	auto b = libssh2_knownhost_readfile(hosts, ~file_path, LIBSSH2_KNOWNHOST_FILE_OPENSSH) >= 0;
	return b ? b : Error();
}

bool KnownHosts::Save()
{
	return SaveAs(file_path);
}

bool KnownHosts::SaveAs(const String& filename)
{
	ASSERT(ssh);
	auto b = hosts && libssh2_knownhost_writefile(hosts, ~filename, LIBSSH2_KNOWNHOST_FILE_OPENSSH) == 0;
	return b ? b : Error();
}

KnownHosts::Info KnownHosts::Check(const String& host, int port)
{
	ASSERT(ssh);
	Info info;
	if(hosts) {
		int	   type   = 0;
		size_t length = 0;
		auto*  p = libssh2_session_hostkey(ssh->GetSession(), &length, &type);
		if(!p) {
			Error();
			info.status = LIBSSH2_KNOWNHOST_CHECK_FAILURE;
			return pick(info);
		}
		info.status = libssh2_knownhost_checkp(
			hosts,
			~host,
			port,
			p,
			length,
			LIBSSH2_KNOWNHOST_TYPE_PLAIN |
			LIBSSH2_KNOWNHOST_KEYENC_RAW,
			NULL
		);
		info.key.Set(p, length);
		switch(type) {
			case LIBSSH2_HOSTKEY_TYPE_RSA:
				info.type = LIBSSH2_KNOWNHOST_KEY_SSHRSA;
				break;
			case LIBSSH2_HOSTKEY_TYPE_DSS:
				info.type = LIBSSH2_KNOWNHOST_KEY_SSHDSS;
				break;
			case LIBSSH2_HOSTKEY_TYPE_UNKNOWN:
				info.type = LIBSSH2_KNOWNHOST_KEY_UNKNOWN;
				break;
			default:
				NEVER();
		}
	}
	return pick(info);
}

Vector<SshHost*> KnownHosts::GetHosts()
{
	ASSERT(ssh);
	Vector<SshHost*> v;
	SshHost *prev = NULL, *next = NULL;
	int rc = libssh2_knownhost_get(hosts, &prev, NULL);
	if(rc >= 0) {
		v.Add(prev);
		if(rc == 0)
			while(rc < 1) {
				rc = libssh2_knownhost_get(hosts, &next, prev);
				if(rc < 0) break;
				v.Add(next);
				prev = next;
			}
	}
	if(rc < 0) Error();
	return pick(v);
}

bool KnownHosts::Error()
{
	String msg = !hosts ? t_("Invalid host handle.") : "";
	hosts_error = ssh_liberror(ssh->GetSession(), hosts ? 0 : -1, msg);
	return false;
}

KnownHosts::KnownHosts(Ssh& session)
{
	ssh   = &session;
	hosts = libssh2_knownhost_init(ssh->GetSession());
}

KnownHosts::~KnownHosts()
{
	if(ssh && hosts)
		libssh2_knownhost_free(hosts);
}
}